/*
 * UNIX shell
 *
 * S. R. Bourne
 * Bell Telephone Laboratories
 */

#include "defs.h"
#include <errno.h>

#define ARGMK	01

extern char *sysmsg[];

static char	*execs(char *ap, char *t[]);
static void	gsort(char *from[], char *to[]);
static int	split(char *s);

/* service routines for `execute' */

void
initio(struct ionod *iop)
{
	char *ion;
	int iof, fd = 0;

	if (iop) {
		iof = iop->iofile;
		ion = mactrim(iop->ioname);
		if (*ion && (flags & noexec) == 0) {
			if (iof & IODOC) {
				subst(chkopen(ion), (fd = tmpfil()));
				close(fd);
				fd = chkopen(tmpout);
				unlink(tmpout);
			} else if (iof & IOMOV) {
				if (eq(minus, ion)) {
					fd = -1;
					close(iof & IOUFD);
				} else if ((fd = stoi(ion)) >= USERIO)
					failed(ion, badfile);
				else {
					fd = dup(fd);
					if (fd < 0)
						perror("dup");
				}
			} else if ((iof & IOPUT) == 0)
				fd = chkopen(ion);
			else if (flags & rshflg)
				failed(ion, restricted);
			else if (iof & IOAPP && (fd = open(ion, 1)) >= 0)
				lseek(fd, 0L, 2);
			else
				fd = create(ion);
			if (fd >= 0)
				myrename(fd, iof & IOUFD);
		}
		initio(iop->ionxt);
	}
}

char *
getpath(char *s)
{
	char *path;

	if (any('/', s)) {
		if (flags & rshflg)
			failed(s, restricted);
		else
			return nullstr;
	} else if ((path = pathnod.namval) == 0)
		return defpath;
	else
		return cpystak(path);
	return 0;
}

int
pathopen(char *path, char *name)
{
	UFD f;

	do {
		path = catpath(path, name);
	} while ((f = open(curstak(), 0)) < 0 && path);
	return f;
}

/*
 * leaves result on top of stack
 */
char *
catpath(char *path, char *name)
{
	char *scanp = path;

	usestak();
	/* push first component of path */
	while (*scanp && *scanp != COLON)
		pushstak(*scanp++);
	if (scanp != path)		/* first char wasn't ':', */
		pushstak('/');
	path = (*scanp? scanp+1: 0);	/* reset path to just after ':' */

	/* append name by pushing it */
	scanp = name;
	do {
		pushstak(*scanp);
	} while (*scanp++);

	setstak(0);
	return path;
}

static char *xecmsg;
static char **xecenv;

void
execa(char *t[])
{
	char *path;

	if ((flags & noexec) == 0) {
		xecmsg = notfound;
		path = getpath(*t);
		namscan(exname);
		xecenv = shsetenv();
		while (path = execs(path, t))
			continue;
		failed(*t, xecmsg);
	}
}

static char *
execs(char *ap, char *t[])
{
	char *p, *prefix;

	prefix = catpath(ap, t[0]);
	trim(p = curstak());

	sigchk();
	execve(p, &t[0], xecenv);

	/* exec failed.  figure out why */
	switch (errno) {
	case ENOEXEC:
		flags = 0;
		comdiv = 0;
		ioset = 0;
		clearup();	/* remove open files and for loop junk */
		if (input)
			close(input);
		close(output);
		output = 2;
		input = chkopen(p);

		/* set up new args */
		setargs(t);
		longjmp(subshell, 1);
		/* not reached */

	case ENOMEM:
		failed(p, toobig);

	case E2BIG:
		failed(p, arglist);

	case ETXTBSY:
		failed(p, txtbsy);

	default:
		xecmsg = badexec;
		/* fall through */
	case ENOENT:
		return prefix;
	}
}

/* for processes to be waited for */
#define MAXP 60
static int pwlist[MAXP];
static int pwc;

void
postclr(void)
{
	int *pw = pwlist;

	while (pw <= &pwlist[pwc])
		*pw++ = 0;
	pwc = 0;
}

void
post(int pcsid)
{
	int *pw = pwlist;

	if (pcsid) {
		while (*pw)
			pw++;
		if (pwc >= MAXP - 1)
			pw--;
		else
			pwc++;
		*pw = pcsid;
	}
}

void
await(int i)
{
	int p, rc = 0, sig, wx = 0, w, ipwc = pwc, w_hi;
	int *pw;

	post(i);
	while (pwc) {
		pw = pwlist;
		p = wait(&w);
		while (pw <= &pwlist[ipwc])
			if (*pw == p) {
				*pw = 0;
				pwc--;
			} else
				pw++;
		if (p == -1)
			continue;

		w_hi = (w >> 8) & LOBYTE;
		if (sig = w & 0177) {
			if (sig == 0177	/* ptrace! return */) {
				prs("ptrace: ");
				sig = w_hi;
			}
			if (sysmsg[sig]) {
				if (i != p || (flags & prompt) == 0) {
					prp();
					prn(p);
					blank();
				}
				prs(sysmsg[sig]);
				if (w & 0200)
					prs(coredump);
			}
			newline();
		}

		if (rc == 0)
			rc = (sig? sig | SIGFLG: w_hi);
		wx |= w;
	}

	if (wx && flags & errflg)
		exitsh(rc);
	exitval = rc;
	exitset();
}

BOOL nosubst;

void
trim(char *p)
{
	char c, q = 0;

	if (p)
		while (c = *p) {
			*p++ = c & STRIP;
			q |= c;
		}
	nosubst = q & QUOTE;
}

char *
mactrim(char *s)
{
	char *t = macro(s);

	trim(t);
	return t;
}

char **
scan(int argn)
{
	struct argnod *argp = (struct argnod *)((uintptr_t)gchain & ~ARGMK);
	char **comargn, **comargm;

	comargn = (char **)getstak(BYTESPERWORD * (argn + 1));
	comargm = comargn += argn;
	*comargn = ENDARGS;

	while (argp) {
		*--comargn = argp->argval;
		if (argp = argp->argnxt)
			trim(*comargn);
		if (argp == 0 || (uintptr_t)argp & ARGMK) {
			gsort(comargn, comargm);
			comargm = comargn;
		}
		/* Lcheat(argp) &= ~ARGMK; */
		argp = (struct argnod *)((uintptr_t)argp & ~ARGMK);
	}
	return comargn;
}

static void
gsort(char *from[], char *to[])		/* a Shell sort */
{
	int i, j, k, m, n;
	char *s;
	char **fromi;

	if ((n = to - from) <= 1)
		return;
	for (j = 1; j <= n; j *= 2)
		;
	for (m = 2 * j - 1; m /= 2;) {
		k = n - m;
		for (j = 0; j < k; j++)
			for (i = j; i >= 0; i -= m) {
				fromi = &from[i];
				if (strcmp(fromi[m], fromi[0]) > 0)
					break;
				else {
					s = fromi[m];
					fromi[m] = fromi[0];
					fromi[0] = s;
				}
			}
	}
}

/* Argument list generation */

int
getarg(struct comnod *c)
{
	int count = 0;
	struct argnod *argp;

	if (c)
		for (argp = c->comarg; argp; argp = argp->argnxt)
			count += split(macro(argp->argval));
	return count;
}

static int
split(char *s)		/* blank interpretation routine */
{
	struct argnod *argp;
	int c, count = 0;

	for (;;) {
		sigchk();
		staktop = locstak() + BYTESPERWORD;
		while (c = *s++, !any(c, ifsnod.namval) && c)
			pushstak(c);
		if (BYTESPERWORD == relstak()) {
			if (c)
				continue;
			setstak(0);
			return count;
		} else if (c == 0)
			s--;

		/*
		 * file name generation
		 */
		argp = (struct argnod *)fixstak();
		if ((c = expand(argp->argval, 0)))
			count += c;
		else {
			makearg(argp);
			count++;
		}
		gchain = (struct argnod *)((uintptr_t)gchain | ARGMK);
	}
}
